import React, { useState, useEffect } from "react";
import "./App.css";
import {
  StarPrinterFactory,
  InterfaceType,
  StarConnectionSettings,
  StarIO10Logger,
} from "star-io10-web";

function Status() {
  const [identifier, setIdentifier] = useState<string>("");
  const [printerFactory, setPrinterFactory] = useState<StarPrinterFactory>(
    () => {
      console.log("create printer factory");
      return new StarPrinterFactory();
    },
  );
  const handleBeforeUnloadEvent = (event: BeforeUnloadEvent) => {
    showMessage("dispose printer factory");
    printerFactory.dispose();
  };
  useEffect(() => {
    // when the window is closed, dispose the printer factory.
    window.addEventListener("beforeunload", handleBeforeUnloadEvent, true);
    return () =>
      window.removeEventListener("beforeunload", handleBeforeUnloadEvent, true);
  }, []);

  const [isProcessing, setIsProcessing] = useState<boolean>(false);
  const [message, setMessage] = useState<string>("");
  async function showMessage(msg: string) {
    console.log(msg);
    setMessage((prev: string) => prev + msg + "\n");
  }

  async function onPressGetStatusButton() {
    showMessage("on press get status button");
    setIsProcessing(true);

    // Start debug log output to console
    // StarIO10Logger.start();

    const connectionSettings = new StarConnectionSettings();
    connectionSettings.interfaceType = InterfaceType.Usb;
    if (identifier.length > 0) {
      connectionSettings.identifier = identifier;
    }

    const printer = printerFactory.createStarPrinter(connectionSettings);
    // const printer = new StarPrinter(connectionSettings);
    // `new StarPrinter(connectionSettings)` may take a long time to process.
    // If you want to separate the process to create StarPrinter, use StarPrinterFactory.

    try {
      await printer.open();
      showMessage("open success");

      const status = await printer.getStatus();
      showMessage(`
get status success

    Has Error: ${status.hasError}
    Paper Empty: ${status.paperEmpty}
    Paper Near Empty: ${status.paperNearEmpty}
    Cover Open: ${status.coverOpen}
    Drawer Open Close Signal: ${status.drawerOpenCloseSignal}
`);

      await printer.close();
      showMessage("close success");
    } catch (error) {
      try {
        await printer?.close();
      } catch {
        // do nothing.
      }

      showMessage("get status failed " + error);
      console.error(error);
    } finally {
      // StarIO10Logger.stop();
      await printer.dispose();
      setIsProcessing(false);
    }
  }

  async function onPressGetInformationButton() {
    showMessage("on press get information button");
    setIsProcessing(true);

    // Start debug log output to console
    // StarIO10Logger.start();

    const connectionSettings = new StarConnectionSettings();
    connectionSettings.interfaceType = InterfaceType.Usb;
    if (identifier.length > 0) {
      connectionSettings.identifier = identifier;
    }

    const printer = printerFactory.createStarPrinter(connectionSettings);
    // const printer = new StarPrinter(connectionSettings);
    // `new StarPrinter(connectionSettings)` may take a long time to process.
    // If you want to separate the process to create StarPrinter, use StarPrinterFactory.

    try {
      await printer.open();
      showMessage("open success");

      const info = await printer.getInformation();
      showMessage(`
get information success

    Model Name: ${info.model}
    Emulation: ${info.emulation}
`);

      await printer.close();
      showMessage("close success");
    } catch (error) {
      try {
        await printer?.close();
      } catch {
        // do nothing.
      }

      showMessage("get information failed " + error);
      console.error(error);
    } finally {
      // StarIO10Logger.stop();
      await printer.dispose();
      setIsProcessing(false);
    }
  }

  return (
    <div className="Get Status">
      {isProcessing && (
        <div className="App-overlay">
          <div className="App-overlay-message">...</div>
        </div>
      )}
      <div>
        <label>Identifier: </label>
        <input
          value={identifier}
          onChange={(e) => setIdentifier(e.target.value)}
        />
        <br />
        <button id="GetStatus" onClick={onPressGetStatusButton}>
          Get Status
        </button>
        <br />
        <button id="GetInformation" onClick={onPressGetInformationButton}>
          Get Information
        </button>
        <br />
        <div style={{ whiteSpace: "pre-line" }}>
          <p>{message}</p>
        </div>
      </div>
    </div>
  );
}

export default Status;
